document.getElementById('extractBtn').addEventListener('click', async () => {
  const statusDiv = document.getElementById('status');
  statusDiv.textContent = "Extracting...";
  statusDiv.style.color = "#2563eb";

  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    // 1. Execute script to extract data
    chrome.scripting.executeScript({
      target: { tabId: tab.id },
      function: extractProductData,
    }, async (results) => {
      if (chrome.runtime.lastError) {
        statusDiv.textContent = "Error: " + chrome.runtime.lastError.message;
        statusDiv.style.color = "red";
        return;
      }

      const data = results[0].result;
      if (data) {
        statusDiv.textContent = "Data extracted! Opening Workbench...";
        statusDiv.style.color = "green";

        // 2. Encode data for URL transport
        // Base64 is safer for URL parameters than raw JSON
        const jsonString = JSON.stringify(data);
        const encodedData = encodeURIComponent(jsonString);

        // 3. Open the App with data in Query Param
        // Using localhost as requested. Change to production URL when deploying.
        const appUrl = `http://uoryn.online/app?import=${encodedData}`;

        chrome.tabs.create({ url: appUrl });

      } else {
        statusDiv.textContent = "Could not extract data.";
        statusDiv.style.color = "red";
      }
    });
  } catch (e) {
    statusDiv.textContent = "Error: " + e.message;
    statusDiv.style.color = "red";
  }
});

// This function runs in the context of the page
function extractProductData() {
  const getMeta = (name) => {
    const el = document.querySelector(`meta[name="${name}"], meta[property="${name}"]`);
    return el ? el.content : "";
  };

  const getSelector = (sel) => {
    const el = document.querySelector(sel);
    return el ? el.innerText.trim() : "";
  };

  const getSrc = (sel) => {
    const el = document.querySelector(sel);
    return el ? el.src : "";
  };

  // --- Platform Specific Extraction ---

  // Amazon
  const isAmazon = window.location.hostname.includes("amazon");
  let amazonTitle = getSelector("#productTitle");
  let amazonDesc = getSelector("#productDescription") || getSelector("#feature-bullets");
  let amazonImg = getSelector("#landingImage") ? document.querySelector("#landingImage").src : "";
  let amazonPrice = getSelector(".a-price .a-offscreen") || getSelector(".a-price .a-price-whole");

  // TikTok Shop
  const isTikTok = window.location.hostname.includes("tiktok");
  // TikTok selectors are dynamic and obfuscated, relying on Meta/OG is best

  // Shopify (Generic)
  const isShopify = window.Shopify || document.querySelector('script[src*="shopify"]');

  // Etsy
  const isEtsy = window.location.hostname.includes("etsy");
  let etsyTitle = getSelector("h1[data-buy-box-listing-title]");
  let etsyPrice = getSelector("p.wt-text-title-03");

  // Shopee
  const isShopee = window.location.hostname.includes("shopee");

  // Lazada
  const isLazada = window.location.hostname.includes("lazada");
  let lazadaTitle = getSelector(".pdp-mod-product-badge-title");
  let lazadaImg = getSrc(".pdp-mod-common-image");
  let lazadaPrice = getSelector(".pdp-price_type_normal");

  // --- Consolidation ---

  let title =
    (isAmazon && amazonTitle) ? amazonTitle :
      (isEtsy && etsyTitle) ? etsyTitle :
        (isLazada && lazadaTitle) ? lazadaTitle :
          getMeta("og:title") ||
          getMeta("twitter:title") ||
          getSelector("h1") ||
          document.title;

  let description =
    (isAmazon && amazonDesc) ? amazonDesc :
      getMeta("og:description") ||
      getMeta("twitter:description") ||
      getSelector("#productDescription") ||
      getSelector(".product-description") ||
      getSelector(".description") ||
      "";

  let image =
    (isAmazon && amazonImg) ? amazonImg :
      (isLazada && lazadaImg) ? lazadaImg :
        getMeta("og:image") ||
        getMeta("twitter:image") ||
        (document.querySelector("img") ? document.querySelector("img").src : "");

  let price =
    (isAmazon && amazonPrice) ? amazonPrice :
      (isEtsy && etsyPrice) ? etsyPrice :
        (isLazada && lazadaPrice) ? lazadaPrice :
          getSelector(".price") ||
          getSelector(".amount") ||
          "";

  // Clean up
  if (title) title = title.trim();
  if (description) description = description.trim();

  return {
    title,
    description,
    image,
    price,
    url: window.location.href,
    source: window.location.hostname,
    platform: isAmazon ? "Amazon" : isTikTok ? "TikTok Shop" : isEtsy ? "Etsy" : isShopee ? "Shopee" : isLazada ? "Lazada" : isShopify ? "Shopify" : "Other"
  };
}