// Listen for messages from the popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === "extractData") {
        const data = extractProductData();
        sendResponse(data);
    }
});

function extractProductData() {
    const hostname = window.location.hostname;
    let data = {
        title: "",
        description: "",
        price: "",
        images: [],
        url: window.location.href,
        platform: "Generic",
        currency: "$",
        hashtags: [],
        category: ""
    };

    try {
        if (hostname.includes("amazon")) {
            extractAmazon(data);
        } else if (hostname.includes("tiktok")) {
            extractTikTok(data);
        } else if (hostname.includes("lazada")) {
            extractLazada(data);
        } else if (hostname.includes("shopee")) {
            extractShopee(data);
        } else if (hostname.includes("shopify") || document.querySelector('script[src*="shopify"]')) {
            extractShopify(data);
        } else if (hostname.includes("etsy")) {
            extractEtsy(data);
        } else {
            extractGeneric(data);
        }
    } catch (e) {
        console.error("ListingMorph Extraction Error:", e);
    }

    // Fallbacks
    if (!data.title) data.title = document.title;
    if (data.images.length === 0) {
        const ogImg = document.querySelector('meta[property="og:image"]')?.content;
        if (ogImg) data.images.push(ogImg);
    }
    
    // Clean up
    data.title = (data.title || "").trim();
    data.description = (data.description || "").trim();
    data.price = (data.price || "").replace(/[^\d.,]/g, "").trim(); // Keep only numbers and dots/commas
    // Remove duplicate images and invalid ones
    data.images = [...new Set(data.images.filter(img => img && !img.includes('sprite') && !img.includes('icon') && !img.includes('data:image')))];

    return data;
}

function extractAmazon(data) {
    data.platform = "Amazon";
    data.title = getText("#productTitle");
    
    // Bullet points are crucial for Amazon description
    const bullets = Array.from(document.querySelectorAll("#feature-bullets li span"))
        .map(el => el.innerText.trim())
        .filter(t => t);
    data.description = bullets.join("\n");
    
    // Price
    data.price = getText(".a-price .a-offscreen") || getText("#price_inside_buybox");
    
    // Images
    const script = document.body.innerHTML.match(/colorImages':\s*({.*?}),/);
    if (script && script[1]) {
        try {
            const imgData = JSON.parse(script[1]);
            if (imgData.initial) {
                imgData.initial.forEach(item => {
                    if (item.hiRes) data.images.push(item.hiRes);
                    else if (item.large) data.images.push(item.large);
                });
            }
        } catch(e) {}
    }
    
    if (data.images.length === 0) {
        const img = document.querySelector("#landingImage");
        if (img) data.images.push(img.src);
    }
}

function extractTikTok(data) {
    data.platform = "TikTok Shop";
    // TikTok Shop structure is complex and dynamic, often obfuscated classes.
    // Try meta tags first as they are reliable
    data.title = getMeta("og:title");
    data.description = getMeta("og:description");
    data.price = getMeta("product:price:amount");
    
    // Try to find images in the gallery container
    document.querySelectorAll('img[class*="swiper-slide"], .product-gallery img').forEach(img => {
        if (img.src && img.width > 200) data.images.push(img.src);
    });
}

function extractLazada(data) {
    data.platform = "Lazada";
    data.title = getText(".pdp-mod-product-badge-title");
    data.price = getText(".pdp-price");
    
    // Description is often in a JSON blob or specific container
    // Lazada puts description in a script tag sometimes
    const descEl = document.querySelector('.pdp-product-detail');
    if (descEl) data.description = descEl.innerText;
    
    // Images
    document.querySelectorAll('.pdp-mod-common-image').forEach(img => {
        if (img.src) data.images.push(img.src.replace(/_\d+x\d+.*$/, '')); // Get full res
    });
}

function extractShopee(data) {
    data.platform = "Shopee";
    // Shopee heavily relies on shadow DOM or canvas sometimes, but meta usually works
    data.title = getMeta("og:title"); // Often formatted "Shop [Product] at Shopee"
    if (data.title && data.title.includes("Shopee")) {
        // Try to clean it
        data.title = data.title.split("|")[0].trim();
    }
    
    data.description = getMeta("og:description");
    
    // Images
    const bgImages = Array.from(document.querySelectorAll('div[style*="background-image"]'));
    bgImages.forEach(div => {
        const match = div.style.backgroundImage.match(/url\("?(.*?)"?\)/);
        if (match && match[1]) data.images.push(match[1]);
    });
}

function extractShopify(data) {
    data.platform = "Shopify";
    // Shopify sites usually have a global `meta` object or standard classes
    try {
        // Check for JSON-LD
        const ldScript = document.querySelector('script[type="application/ld+json"]');
        if (ldScript) {
            const ld = JSON.parse(ldScript.innerText);
            if (ld) {
                data.title = ld.name;
                data.description = ld.description;
                data.price = ld.offers?.price || ld.offers?.[0]?.price;
                if (ld.image) {
                     if (Array.isArray(ld.image)) data.images = ld.image;
                     else data.images.push(ld.image);
                }
            }
        }
    } catch(e) {}
    
    if (!data.title) data.title = getText(".product-single__title, .product__title, h1");
}

function extractEtsy(data) {
    data.platform = "Etsy";
    data.title = getText("h1.wt-text-body-03");
    data.price = getText("p.wt-text-title-03");
    data.description = getText("#product-details-content-toggle");
    
    document.querySelectorAll(".carousel-pane img").forEach(img => {
        if (img.src && img.src.includes("il_")) {
            data.images.push(img.src.replace("il_75x75", "il_fullxfull").replace("il_340x270", "il_fullxfull"));
        }
    });
}

function extractGeneric(data) {
    data.title = getMeta("og:title") || document.title;
    data.description = getMeta("og:description");
    data.price = getMeta("product:price:amount");
}

// Helpers
function getText(selector) {
    const el = document.querySelector(selector);
    return el ? el.innerText.trim() : "";
}

function getMeta(property) {
    const el = document.querySelector(`meta[property="${property}"]`) || document.querySelector(`meta[name="${property}"]`);
    return el ? el.content : "";
}
